/*
********************************************************************************
*
*   Copyright (c) 2017 by Industrial Control Communications, Inc.
*
*   This software is copyrighted by and is the sole property of
*   Industrial Control Communications, Inc.  Any unauthorized use,
*   duplication, transmission, distribution, or disclosure of this
*   software is expressly forbidden.
*
*   This Copyright notice may not be removed or modified without prior
*   written consent of Industrial Control Communications, Inc.
*
*   ICC, Inc.
*   230 Horizon Dr                      	USA 608.831.1255
*   Suite 100                               http://www.iccdesigns.com
*   Verona, WI 53593                     	support@iccdesigns.com
*
********************************************************************************
*
*          Project: SpiLinuxExample
*        File Name: IccSpiProtocol.c
*    Original Date: 08.24.2017
*           Author: Josh Schulze
*
*      Description: This file contains functions to build ICC SPI request
*      				and process the response.
*
*
*   Edit Date           Edit Description
*   ===============     ========================================================
*
*
********************************************************************************
*/


/*
********************************************************************************
*                                   INCLUDES
********************************************************************************
*/

#include "IccSpiProtocol.h"

/*
********************************************************************************
*                               PUBLIC FUNCTIONS
********************************************************************************
*/

/*
********************************************************************************
* Function    : BuildIccSpiGetStatus
* Description : Populates a Get Status request into a buffer
* Arguments   : uint8_t*	: The buffer to put the request in
* Returns     : None
* Comments    :
********************************************************************************
*/
void BuildIccSpiGetStatus (uint8_t *buffer)
{
	buffer[0] = SPI_CMD_GET_STATUS;
	buffer[1] = 0x00;
	buffer[2] = 0x00;
	buffer[3] = 0x00;
	buffer[4] = 0x00;
}

/*
********************************************************************************
* Function    : BuildIccSpiSetAddress
* Description : Populates a Set Address request into a buffer
* Arguments   : uint8_t*	: The buffer to put the request in
*				uint16_t	: The address to use
* Returns     : None
* Comments    :
********************************************************************************
*/
void BuildIccSpiSetAddress (uint8_t *buffer, uint16_t address)
{
	buffer[0] = SPI_CMD_SET_ADDR;
	buffer[1] = 0x00;
	buffer[2] = 0x00;
	buffer[3] = (uint8_t)(address >> 8);
	buffer[4] = (uint8_t)address;
}

/*
********************************************************************************
* Function    : BuildIccSpiRead
* Description : Populates a Read request of a given length into a buffer
* Arguments   : uint8_t*	: The buffer to put the request in
*				uint8_t		: The number of bytes to read
*						  	  This must be either 1, 2, or 4
* Returns     : None
* Comments    :
********************************************************************************
*/
void BuildIccSpiRead (uint8_t *buffer, uint8_t numBytes)
{
	switch(numBytes)
	{
		case 1:
			buffer[0] = SPI_CMD_READ_BYTE;
			break;
		case 2:
			buffer[0] = SPI_CMD_READ_SHORT;
			break;
		default:
			buffer[0] = SPI_CMD_READ_LONG;
			break;
	}

	buffer[1] = 0x00;
	buffer[2] = 0x00;
	buffer[3] = 0x00;
	buffer[4] = 0x00;
}

/*
********************************************************************************
* Function    : BuildIccSpiWrite
* Description : Populates a Write request of a given length into a buffer
* Arguments   : uint8_t*	: The buffer to put the request in
*				uint8_t		: The number of bytes to write
*						  	  This must be either 1, 2, or 4
*				uint32_t	: The data to write
* Returns     : None
* Comments    :
********************************************************************************
*/
void BuildIccSpiWrite (uint8_t *buffer, uint8_t numBytes, uint32_t writeData)
{
	buffer[1] = 0x00;
	buffer[2] = 0x00;
	buffer[3] = 0x00;
	buffer[4] = 0x00;

	switch(numBytes)
	{
		case 1:
			buffer[0] = SPI_CMD_WRITE_BYTE;
			buffer[4] = (uint8_t)writeData;
			break;
		case 2:
			buffer[0] = SPI_CMD_WRITE_SHORT;
			buffer[3] = (uint8_t)(writeData >> 8);
			buffer[4] = (uint8_t)writeData;
			break;
		default:
			buffer[0] = SPI_CMD_WRITE_LONG;
			buffer[1] = (uint8_t)(writeData >> 24);
			buffer[2] = (uint8_t)(writeData >> 16);
			buffer[3] = (uint8_t)(writeData >> 8);
			buffer[4] = (uint8_t)writeData;
			break;
	}
}

/*
********************************************************************************
* Function    : GetIccSpiResponseData
* Description : Retrieves the response data from an SPI packet
* Arguments   : uint8_t*		: The buffer to get the data from
* 				SpiStatusReg*	: Points to the location to store the SPI status register
* 				uint32_t*		: Points to the location to store the data
* Returns     : None
* Comments    :
********************************************************************************
*/
void GetIccSpiResponseData (uint8_t *buffer, SpiStatusReg *status, uint32_t *data)
{
	*status = *((SpiStatusReg*)&buffer[0]);

	*data = 0;
	*data |= (((uint32_t)buffer[1]) << 24);
	*data |= (((uint32_t)buffer[2]) << 16);
	*data |= (((uint32_t)buffer[3]) << 8);
	*data |= (uint32_t)buffer[4];
}
