/*
********************************************************************************
*	
*   Copyright (c) 2017 by Industrial Control Communications, Inc.
*
*   This software is copyrighted by and is the sole property of
*   Industrial Control Communications, Inc.  Any unauthorized use,
*   duplication, transmission, distribution, or disclosure of this
*   software is expressly forbidden.
*
*   This Copyright notice may not be removed or modified without prior
*   written consent of Industrial Control Communications, Inc.
*
*   ICC, Inc.
*   230 Horizon Dr                      	USA 608.831.1255
*   Suite 100                               http://www.iccdesigns.com
*   Verona, WI 53593                     	support@iccdesigns.com
*
********************************************************************************
*
*          Project: SpiLinuxExample
*        File Name: SpiLinuxExample.c
*    Original Date: 08.24.2017
*           Author: Josh Schulze
*
*      Description: This file contains the main entry point for the SpiLinuxExample
*      				program.
*
*      				This program is used to send SPI commands to a slave device
*      				from commands entered by a user on the command line.
*
*      				Files included in this example:
*      				cpu.h				-	Contains platform-dependent definitions for the
*      						  				SPI device that the program will use.
*
*					IccSpiProtocol.c 	-	Contains definitions and functions for building
*					IccSpiProtocol.h		and parsing ICC SPI messages. These files may
*											be used directly in an OEM's own application.
*
*					SpiCommand.c		-	Linux-specific functions to open the SPI port
*					SpiCommand.h			and send/receive SPI messages. The functions
*											contained in these files may be used directly
*											in an OEM's own Linux-based application.
*
*					UserIO.c			-	Used by this program for interacting with a
*					UserIO.h				a user via the command line.
*
*
*   Edit Date           Edit Description
*   ===============     ========================================================
*   
*
********************************************************************************
*/

/*
********************************************************************************
*                                   INCLUDES
********************************************************************************
*/

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>			// System errors

#include "SpiLinuxExample.h"
#include "SpiCommand.h"
#include "UserIO.h"

/*
********************************************************************************
*                                 LOCAL MACROS
********************************************************************************
*/

/* --- Program Defaults --- */
#define DEFAULT_SPI_SPEED		1500			// SPI speed to use if the user does not enter a valid speed option
#define DEFAULT_SPI_MODE		SPI_MODE_0		// SPI mode to use if the user does not enter a valid SPI mode option

/*
********************************************************************************
*                              PRIVATE PROTOTYPES
********************************************************************************
*/

static void InitializeSpiData (struct spi_ioc_transfer *p_spi_msg, uint8_t *tx_buf, uint8_t *rx_buf, SpiConfig *p_spi_cfg);

static void RunSpiClient (SpiConfig *p_spi_cfg);


/*
********************************************************************************
*                               PUBLIC FUNCTIONS
********************************************************************************
*/

/*
********************************************************************************
* Function    : main
* Description : Main entry point for the program
* Arguments   : int		: the argument count
* 				char*	: the argument values
* Returns     : None
* Comments    :
********************************************************************************
*/
int main (int argc, char *argv[])
{
	SpiConfig   spi_port_config;
	int			opt;
	uint8_t		temp_uint8_t;
	uint8_t		spi_mode	 		= DEFAULT_SPI_MODE;
	uint32_t 	spi_speed_hz 		= DEFAULT_SPI_SPEED;


	if (argc > 1 && strcmp(argv[1], "--help") == 0)
	{
		// Print usage information
		PrintHelp(true);
		exit(EXIT_SUCCESS);
	}

	// Get the command arguments
	do
	{
		opt = getopt(argc, argv, "s:m:");

		switch (opt)
		{
			case 's':
				// get the spi speed
				sscanf(optarg, "%d", &spi_speed_hz);

				if (spi_speed_hz > SPI_MAX_FREQUENCY)
				{
					spi_speed_hz = DEFAULT_SPI_SPEED;
					printf("Invalid SPI speed, defaulted to %d.\n", DEFAULT_SPI_SPEED);
				}
				break;

			case 'm':
				// get SPI mode
				sscanf(optarg, "%hhu", &temp_uint8_t);

				switch (temp_uint8_t)
				{
					case 0:
						spi_mode = SPI_MODE_0;
						break;

					case 1:
						spi_mode = SPI_MODE_1;
						break;

					case 2:
						spi_mode = SPI_MODE_2;
						break;

					case 3:
						spi_mode = SPI_MODE_3;
						break;

					default:
						spi_mode = DEFAULT_SPI_MODE;
						printf("Invalid SPI mode, defaulted to mode %d.\n", DEFAULT_SPI_MODE);
						break;
				}
				break;

			case EOF:
				break;

			default:	// '?'
				printf("Try 'SpiLinuxExample --help' for more information.\n");
				exit(EXIT_SUCCESS);
				break;
		}
	} while (opt != EOF);

	// Setup SPI Port Configuration
	spi_port_config.mode 			= spi_mode;
	spi_port_config.lsb_first		= false;
	spi_port_config.bits_per_word	= 8;
	spi_port_config.speed_hz 		= spi_speed_hz;

	// Print settings if not stress testing
	PrintSpiPortSettings(&spi_port_config);

	// Start the SPI Client
	RunSpiClient(&spi_port_config);

	return EXIT_SUCCESS;
}

/*
********************************************************************************
*                               PRIVATE FUNCTIONS
********************************************************************************
*/

/*
********************************************************************************
* Function    : InitializeSpiData
* Description : Initializes the spi_ioc_transfer structure passed in.
* Arguments   : struct spi_ioc_transfer*	: the transfer structure to initialize
* 				uint8_t*					: the transmit buffer
* 				uint8_t*					: the receive buffer
* 				SpiConfig*					: the SPI port configuration
* Returns     : None
* Comments    :
********************************************************************************
*/
static void InitializeSpiData (struct spi_ioc_transfer *p_spi_msg, uint8_t *tx_buf, uint8_t *rx_buf, SpiConfig *p_spi_cfg)
{
	bzero(p_spi_msg, sizeof(struct spi_ioc_transfer));

	// Set up SPI transfer structure
	p_spi_msg->tx_buf 			= (unsigned long)tx_buf;
	p_spi_msg->rx_buf 			= (unsigned long)rx_buf;
	p_spi_msg->len 				= SPI_PACKET_LEN;
	p_spi_msg->speed_hz			= p_spi_cfg->speed_hz;
	p_spi_msg->delay_usecs		= (1000000 / p_spi_cfg->speed_hz) + 1;		// PicoPort requires a THold of at least 1 clock period - always round up 1us
	p_spi_msg->bits_per_word	= p_spi_cfg->bits_per_word;
	p_spi_msg->cs_change		= false;									// always deselect the PicoPort's chip select after each transaction
}

/*
********************************************************************************
* Function    : RunSpiClient
* Description : Main loop for SPI communications controlled by the user.
* Arguments   : SpiConfig*	: the SPI port configuration
* Returns     : None
* Comments    :
********************************************************************************
*/
static void RunSpiClient (SpiConfig *p_spi_cfg)
{
	bool 			has_more_cmds;
	bool			cmd_valid;

	// SPI variables
	int 			fd_spi;
	struct spi_ioc_transfer spi_msg;
	uint8_t 		tx_buf[SPI_PACKET_LEN];
	uint8_t 		rx_buf[SPI_PACKET_LEN];

	// Command variables
	SpiCommand 		usr_cmd;


	// Initialize the SPI message structure
	InitializeSpiData(&spi_msg, tx_buf, rx_buf, p_spi_cfg);

	// Open a connection to the SPI device
	fd_spi = OpenSpiConnection(p_spi_cfg);

	// make sure the connection was opened successfully
	if (fd_spi >= 0)
	{
		// Main loop
		while (has_more_cmds)
		{
			// Reset the buffers
			bzero(tx_buf, SPI_PACKET_LEN);
			bzero(rx_buf, SPI_PACKET_LEN);
			bzero(&usr_cmd, sizeof(SpiCommand));

			// Get the command
			has_more_cmds = GetSpiCommandFromUser(&usr_cmd, &cmd_valid);

			if (cmd_valid && has_more_cmds)
			{
				SendCommand(fd_spi, &spi_msg, &usr_cmd);
			}
		}

		// Close the handle
		if (fd_spi > 0)
		{
			close(fd_spi);
		}
	}
}
